<?php
/**
 * Permission Modal Helper
 * Provides reusable permission modal functionality for all pages
 */

require_once __DIR__ . '/../auth/auth.php';

function checkPermissionWithModal($permission, $pageName, $redirectUrl = null) {
    global $pdo, $isSuperAdmin;
    
    // Check if user has permission
    $hasPermission = hasPermission($permission) || $isSuperAdmin;
    $userName = $_SESSION['username'] ?? 'User';
    
    // If no permission, show modal instead of redirecting
    if (!$hasPermission) {
        // Set variables for the modal
        $GLOBALS['showPermissionModal'] = true;
        $GLOBALS['permissionPageName'] = $pageName;
        $GLOBALS['permissionUserName'] = $userName;
        $GLOBALS['permissionRedirectUrl'] = $redirectUrl;
        
        return false;
    }
    
    return true;
}

function getPermissionModalHTML() {
    if (!isset($GLOBALS['showPermissionModal']) || !$GLOBALS['showPermissionModal']) {
        return '';
    }
    
    $pageName = $GLOBALS['permissionPageName'] ?? 'this page';
    $userName = $GLOBALS['permissionUserName'] ?? 'User';
    $redirectUrl = $GLOBALS['permissionRedirectUrl'] ?? 'dashboard.php';
    
    return '
    <!-- Permission Access Modal -->
    <div class="modal fade" id="permissionModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-warning text-dark">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Access Restricted
                    </h5>
                </div>
                <div class="modal-body text-center">
                    <div class="mb-4">
                        <i class="fas fa-user-lock fa-3x text-warning mb-3"></i>
                        <h4 class="text-dark">Hello, ' . htmlspecialchars($userName) . '!</h4>
                        <p class="text-muted">You don\'t have permission to access ' . htmlspecialchars($pageName) . '.</p>
                        <p class="text-info"><strong>Please contact Super Admin for Permission</strong></p>
                    </div>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Redirecting you to a page you have access to...</strong>
                    </div>
                </div>
                <div class="modal-footer justify-content-center">
                    <button type="button" class="btn btn-primary" onclick="redirectToAccessiblePage()">
                        <i class="fas fa-arrow-right me-2"></i>
                        Continue
                    </button>
                </div>
            </div>
        </div>
    </div>';
}

function getPermissionModalCSS() {
    return '
    <style>
    /* Permission Modal Blur Effect */
    .modal-blur .wrapper {
        filter: blur(8px);
        transition: filter 0.3s ease;
    }

    .modal-blur .main-content {
        filter: blur(8px);
        transition: filter 0.3s ease;
    }

    .modal-blur .container-fluid {
        filter: blur(8px);
        transition: filter 0.3s ease;
    }

    .modal-blur .container {
        filter: blur(8px);
        transition: filter 0.3s ease;
    }

    /* Ensure modal and its backdrop are not blurred */
    .modal {
        filter: none !important;
    }

    .modal-backdrop {
        filter: none !important;
    }

    .modal-dialog {
        filter: none !important;
    }

    .modal-content {
        filter: none !important;
    }

    /* Permission modal specific styling */
    #permissionModal .modal-content {
        border: none;
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        backdrop-filter: blur(10px);
        background: rgba(255, 255, 255, 0.95);
    }

    #permissionModal .modal-header {
        border-bottom: 1px solid rgba(0, 0, 0, 0.1);
        border-radius: 20px 20px 0 0;
    }

    #permissionModal .modal-footer {
        border-top: 1px solid rgba(0, 0, 0, 0.1);
        border-radius: 0 0 20px 20px;
    }

    /* Override Bootstrap modal backdrop for better blur effect */
    .modal-backdrop.show {
        opacity: 0.3;
    }
    </style>';
}

function getPermissionModalJS() {
    if (!isset($GLOBALS['showPermissionModal']) || !$GLOBALS['showPermissionModal']) {
        return '';
    }
    
    $redirectUrl = $GLOBALS['permissionRedirectUrl'] ?? 'dashboard.php';
    
    return '
    <script>
    // Permission Modal Functions
    function showPermissionModal() {
        console.log("Attempting to show permission modal...");
        const modalElement = document.getElementById("permissionModal");
        
        if (modalElement) {
            console.log("Modal element found, showing modal...");
            const permissionModal = new bootstrap.Modal(modalElement);
            
            // Add blur effect to background elements
            const wrapper = document.querySelector(".wrapper");
            const mainContent = document.querySelector(".main-content");
            const container = document.querySelector(".container-fluid") || document.querySelector(".container");
            
            if (wrapper) wrapper.classList.add("modal-blur");
            if (mainContent) mainContent.classList.add("modal-blur");
            if (container) container.classList.add("modal-blur");
            
            console.log("Blur classes added to background elements");
            
            permissionModal.show();
            console.log("Permission modal shown successfully");
            
            // Remove blur when modal is hidden
            modalElement.addEventListener("hidden.bs.modal", function() {
                console.log("Modal hidden, removing blur...");
                if (wrapper) wrapper.classList.remove("modal-blur");
                if (mainContent) mainContent.classList.remove("modal-blur");
                if (container) container.classList.remove("modal-blur");
                console.log("Blur classes removed");
            });
        } else {
            console.error("Modal element not found!");
        }
    }

    function redirectToAccessiblePage() {
        console.log("Redirecting user to accessible page...");
        window.location.href = "' . $redirectUrl . '";
    }

    // Show modal when page loads
    document.addEventListener("DOMContentLoaded", function() {
        console.log("DOMContentLoaded - Permission modal should show");
        
        // Apply blur immediately
        const wrapper = document.querySelector(".wrapper");
        const mainContent = document.querySelector(".main-content");
        const container = document.querySelector(".container-fluid") || document.querySelector(".container");
        
        if (wrapper) wrapper.classList.add("modal-blur");
        if (mainContent) mainContent.classList.add("modal-blur");
        if (container) container.classList.add("modal-blur");
        
        console.log("Blur applied immediately on page load");
        
        showPermissionModal();
    });

    // Also try on window load as backup
    window.addEventListener("load", function() {
        console.log("Window load - Checking if modal needs to be shown...");
        const modalElement = document.getElementById("permissionModal");
        if (modalElement && !modalElement.classList.contains("show")) {
            console.log("Modal not shown yet, showing now...");
            
            // Ensure blur is applied
            const wrapper = document.querySelector(".wrapper");
            const mainContent = document.querySelector(".main-content");
            const container = document.querySelector(".container-fluid") || document.querySelector(".container");
            
            if (wrapper) wrapper.classList.add("modal-blur");
            if (mainContent) mainContent.classList.add("modal-blur");
            if (container) container.classList.add("modal-blur");
            
            showPermissionModal();
        }
    });
    </script>';
}

// Always output a generic modal for JS use
function getUniversalPermissionModalHTML() {
    return '
    <!-- Universal Permission Modal for JS-triggered actions -->
    <div class="modal fade" id="universalPermissionModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-warning text-dark">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Access Restricted
                    </h5>
                </div>
                <div class="modal-body text-center">
                    <div class="mb-4">
                        <i class="fas fa-user-lock fa-3x text-warning mb-3"></i>
                        <h4 class="text-dark" id="universalPermissionUser"></h4>
                        <p class="text-muted" id="universalPermissionMessage"></p>
                        <p class="text-info"><strong>Please contact Super Admin for Permission</strong></p>
                    </div>
                </div>
                <div class="modal-footer justify-content-center">
                    <button type="button" class="btn btn-primary" data-bs-dismiss="modal">
                        <i class="fas fa-arrow-right me-2"></i>
                        Continue
                    </button>
                </div>
            </div>
        </div>
    </div>';
}

function getUniversalPermissionModalJS() {
    $userName = $_SESSION['username'] ?? 'User';
    $escapedUserName = addslashes($userName);
    
    // Get user permissions for JavaScript with direct database query
    $userPermissions = [];
    if (isset($_SESSION['user_id'])) {
        try {
            global $pdo;
            $stmt = $pdo->prepare("
                SELECT TRIM(p.name) as name
                FROM role_permissions rp
                JOIN permissions p ON rp.permission_id = p.id
                WHERE rp.role_id = (
                    SELECT role_id 
                    FROM users 
                    WHERE id = ?
                )
            ");
            $stmt->execute([$_SESSION['user_id']]);
            $userPermissions = $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (Exception $e) {
            error_log("Error getting user permissions: " . $e->getMessage());
            $userPermissions = [];
        }
    }
    
    $permissionsJson = json_encode($userPermissions);

    // Get super admin status for JavaScript
    $isSuperAdmin = false;
    if (isset($_SESSION['user_id'])) {
        try {
            global $pdo;
            $stmt = $pdo->prepare("SELECT r.is_super_admin FROM roles r JOIN users u ON r.id = u.role_id WHERE u.id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $isSuperAdmin = (bool)$stmt->fetchColumn();
        } catch (Exception $e) {
            error_log("Error checking super admin status: " . $e->getMessage());
            $isSuperAdmin = false;
        }
    }
    $isSuperAdminJS = $isSuperAdmin ? 'true' : 'false';
    
    $script = '<script>';
    $script .= 'const isSuperAdmin = ' . $isSuperAdminJS . ';';
    $script .= 'const userPermissions = ' . $permissionsJson . ';';
    $script .= 'function hasPermissionJS(permission) {';
    $script .= 'if (isSuperAdmin) return true;';
    $script .= 'return userPermissions.includes(permission);';
    $script .= '}';
    $script .= 'function showPermissionModalJS(permissionLabel) {';
    $script .= 'const modalElement = document.getElementById("universalPermissionModal");';
    $script .= 'if (!modalElement) return;';
    $script .= 'document.getElementById("universalPermissionUser").textContent = "Hello, ' . $escapedUserName . '!";';
    $script .= 'document.getElementById("universalPermissionMessage").textContent = "You do not have permission to access " + permissionLabel + ".";';
    $script .= 'const permissionModal = new bootstrap.Modal(modalElement);';
    $script .= 'permissionModal.show();';
    $script .= 'modalElement.addEventListener("hidden.bs.modal", function() {';
    $script .= 'const currentPage = window.location.pathname.split("/").pop();';
    $script .= 'window.location.href = currentPage;';
    $script .= '}, { once: true });';
    $script .= '}';
    $script .= 'function handlePermissionButton(permission, permissionLabel, callback) {';
    $script .= 'if (hasPermissionJS(permission)) {';
    $script .= 'if (callback) callback();';
    $script .= '} else {';
    $script .= 'showPermissionModalJS(permissionLabel);';
    $script .= '}';
    $script .= '}';
    $script .= 'document.addEventListener("DOMContentLoaded", function() {';
    $script .= 'document.querySelectorAll("[data-permission]").forEach(function(element) {';
    $script .= 'element.addEventListener("click", function(e) {';
    $script .= 'const permission = this.getAttribute("data-permission");';
    $script .= 'const permissionLabel = this.getAttribute("data-permission-label") || permission;';
    $script .= 'if (!hasPermissionJS(permission)) {';
    $script .= 'e.preventDefault();';
    $script .= 'e.stopPropagation();';
    $script .= 'showPermissionModalJS(permissionLabel);';
    $script .= 'return false;';
    $script .= '}';
    $script .= '});';
    $script .= '});';
    $script .= '});';
    $script .= '</script>';
    
    return $script;
}
?> 